// Mini CMS JavaScript

document.addEventListener('DOMContentLoaded', function() {
  // ตรวจสอบ URL parameters สำหรับแสดงข้อความ
  const urlParams = new URLSearchParams(window.location.search);
  const deleted = urlParams.get('deleted');

  if (deleted === '1') {
    showNotification('บทความถูกลบสำเร็จแล้ว!', 'success');
    // ลบ parameter ออกจาก URL
    history.replaceState(null, '', window.location.pathname);
  }

  // เพิ่ม character counter สำหรับ textarea
  const textarea = document.getElementById('content');
  if (textarea) {
    addCharacterCounter(textarea);
  }

  // Auto-save draft (optional feature)
  setupAutoSave();

  // Confirm before leaving page with unsaved changes
  setupUnsavedChangesWarning();
});

// ฟังก์ชันแสดงการแจ้งเตือน
function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  notification.className = `notification notification-${type}`;
  notification.textContent = message;
  notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 25px;
        background: ${type === 'success' ? '#28a745' : '#dc3545'};
        color: white;
        border-radius: 10px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.3);
        z-index: 1000;
        animation: slideIn 0.3s ease;
    `;

  document.body.appendChild(notification);

  setTimeout(() => {
    notification.style.animation = 'slideOut 0.3s ease';
    setTimeout(() => notification.remove(), 300);
  }, 3000);
}

// เพิ่มตัวนับจำนวนตัวอักษร
function addCharacterCounter(textarea) {
  const counter = document.createElement('div');
  counter.className = 'character-counter';
  counter.style.cssText = `
        text-align: right;
        font-size: 0.85em;
        color: #999;
        margin-top: 5px;
    `;

  const updateCounter = () => {
    const length = textarea.value.length;
    counter.textContent = `${length.toLocaleString()} ตัวอักษร`;
  };

  textarea.parentNode.appendChild(counter);
  textarea.addEventListener('input', updateCounter);
  updateCounter();
}

// Auto-save draft
function setupAutoSave() {
  const form = document.querySelector('.article-form');
  if (!form) return;

  const inputs = form.querySelectorAll('input, textarea, select');
  let autoSaveTimer;

  inputs.forEach(input => {
    input.addEventListener('input', () => {
      clearTimeout(autoSaveTimer);
      autoSaveTimer = setTimeout(() => {
        saveDraft(form);
      }, 2000); // Auto-save หลังจาก 2 วินาที
    });
  });

  // โหลด draft ที่บันทึกไว้
  loadDraft(form);
}

function saveDraft(form) {
  const formData = new FormData(form);
  const draft = {};

  for (let [key, value] of formData.entries()) {
    draft[key] = value;
  }

  localStorage.setItem('article_draft', JSON.stringify(draft));

  // แสดงสถานะ auto-save
  const saveIndicator = document.getElementById('save-indicator') || createSaveIndicator();
  saveIndicator.textContent = '💾 บันทึกอัตโนมัติแล้ว';
  saveIndicator.style.color = '#28a745';
}

function loadDraft(form) {
  const draft = localStorage.getItem('article_draft');
  if (!draft) return;

  const data = JSON.parse(draft);

  // ถามผู้ใช้ว่าต้องการโหลด draft หรือไม่
  if (confirm('พบข้อมูลที่บันทึกไว้ ต้องการโหลดหรือไม่?')) {
    for (let [key, value] of Object.entries(data)) {
      const input = form.querySelector(`[name="${key}"]`);
      if (input) {
        input.value = value;
      }
    }
  }
}

function createSaveIndicator() {
  const indicator = document.createElement('div');
  indicator.id = 'save-indicator';
  indicator.style.cssText = `
        position: fixed;
        bottom: 20px;
        right: 20px;
        padding: 10px 20px;
        background: white;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        font-size: 0.9em;
    `;
  document.body.appendChild(indicator);
  return indicator;
}

// เตือนก่อนออกจากหน้าถ้ามีการเปลี่ยนแปลง
function setupUnsavedChangesWarning() {
  const form = document.querySelector('.article-form');
  if (!form) return;

  let formChanged = false;
  const inputs = form.querySelectorAll('input, textarea, select');

  inputs.forEach(input => {
    input.addEventListener('change', () => {
      formChanged = true;
    });
  });

  form.addEventListener('submit', () => {
    formChanged = false;
    localStorage.removeItem('article_draft'); // ลบ draft เมื่อ submit
  });

  window.addEventListener('beforeunload', (e) => {
    if (formChanged) {
      e.preventDefault();
      e.returnValue = '';
    }
  });
}

// เพิ่ม CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }

    @keyframes slideOut {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(100%);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);
